// 3D Math Unit
//
// Authors: Maarten Kronberger, Danie Heath
// Description : Contains all the functions and procedure for 3d mathematical calculations
//
// Functions and procedures :
//        Vector : Returns a unit vector from 2 given points
//        Magnitude : Returns the magnitude(size) of a given vector
//        Cross : Returns a perpendicular vector from 2 given vectors by
//                taking the cross product
//        Distance : Returns the distance between two 3D points
//        Normalize : Returns a normalized vector (a vector of length 1)
//        Normal : Returns the normal of a polygon (the direction a polygon is facing)
//        Dot : Returns the dot product of two vectors
//        AngleBetweenVectors : Returns the angle between two vectors
//        ClosestPointOnLine : Returns the closest point on a line to a given point
//        PlaneDistance : Returns the distance between a plane and the origin
//        IntersectionPoint : Returns the intersection point between a line and a plane
//        InsidePolygon : Checks to see if a point is in the bounds of a polygon
//        IntersectedPlane : Checks to see if a line intersects a plane
//        IntersectedPolygon : Checks to see if a line intersects a polygon
//        ClassifySphere : This tells us if a sphere is in front, behind, or intersects a plane
//        EdgeSphereCollision : Checks if the sphere is intersecting any of the edges of the polygon
//        SpherePolygonCollision : Checks to see if a sphere intersects a polygon
//        GetCollisionOffset : Returns the offset to move the center of the sphere off the collided polygon

unit glMath3D;

interface

uses
  glTypes;

function Vector (vPoint1, vPoint2 : TVertex) : TVertex;
function Magnitude (vVector : TVertex) : Single;
function Cross (vVector1, vVector2 : TVertex) : TVertex;
function Distance (vPoint1, vPoint2 : TVertex) : Single;
function Normalize (vVector : TVertex) : TVertex;
function Normal (vPoly : Array of TVertex) : TVertex;
function Dot (vVector1, vVector2 : TVertex) : Single;
function AngleBetweenVectors(vVector1, vVector2 : TVertex) : Single;
function ClosestPointOnLine (vA, vB, vPoint : TVertex) : TVertex;
function PlaneDistance (vNormal, vPoint : TVertex) : Single;
function IntersectionPoint(vNormal : TVertex; vLine : Array of TVertex; Distance : Single) : TVertex;
function InsidePolygon (vIntersection : TVertex; vPolygon : Array of TVertex; VertexCount : Integer) : Boolean;
function IntersectedPlane (vPolygon, vLine : Array of TVertex; var vNormal : TVertex; var OriginDistance : Single) : Boolean;
function IntersectedPolygon (vPolygon, vLine : Array of TVertex; VertexCount : Integer) : Boolean;
function ClassifySphere(var vCenter : TVertex; var vNormal : TVertex; var vPoint : TVertex; Radius : Single; var Distance : Single) : Integer;
function EdgeSphereCollision(var vCenter : TVertex; vPolygon : Array of TVertex; VertexCount : Integer; Radius : Single) : Boolean;
function SpherePolygonCollision(vPolygon : Array of TVertex; var vCenter : TVertex; VertexCount : Integer; Radius : Single) : Boolean;
function GetCollisionOffset (var vNormal : TVertex; Radius, Dist : Single) : TVertex;

const
  MATCH_FACTOR : Extended = 0.9999;

implementation

uses
  Math;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Vector (vPoint1, vPoint2 : TVertex) : TVertex;
//Returns a unit vector from 2 given points

var
  vVector : TVertex;

begin
  //In order to get a vector from 2 given points, we need to subtract the
  //second point from the first point.
  vVector.X := vPoint1.X - vPoint2.X;
  vVector.Y := vPoint1.Y - vPoint2.Y;
  vVector.Z := vPoint1.Z - vPoint2.Z;

  Result := vVector;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Magnitude (vVector : TVertex) : Single;
//Returns the magnitude(size) of a given vector

begin
  Result := sqrt((vVector.X*vVector.X) +
                 (vVector.Y*vVector.Y) +
                 (vVector.Z*vVector.Z));
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Cross (vVector1, vVector2 : TVertex) : TVertex;
//Returns a perpendicular vector from 2 given vectors by taking the cross product

var
  vCross : TVertex;

begin
  // Any 2 given vectors define a plane.  The cross products is a vector that is perpendicular to that plane,
  // which means it points straight out of the plane at a 90 deg angle.

  //The X value for the vector is : (V1.Y * V2.Z) - (V1.Z * V2.Y)
  vCross.X := (vVector1.Y * vVector2.Z) - (vVector1.Z * vVector2.Y);

  //The Y value for the vector is : (V1.Z * V2.X) - (V1.X * V2.Z)
  vCross.Y := (vVector1.Z * vVector2.X) - (vVector1.X * vVector2.Z);

  //The Z value for the vector is : (V1.X * V2.Y) - (V1.Y * V2.X)
  vCross.Z := (vVector1.X * vVector2.Y) - (vVector1.Y * vVector2.X);

  //Return our cross vector
  Result := vCross;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Distance (vPoint1, vPoint2 : TVertex) : Single;
//Returns the distance between two points

begin
  //This is the classic formula
  Result := sqrt ((vPoint2.X - vPoint1.X)*(vPoint2.X - vPoint1.X) +
                  (vPoint2.Y - vPoint1.Y)*(vPoint2.Y - vPoint1.Y) +
                  (vPoint2.Z - vPoint1.Z)*(vPoint2.Z - vPoint1.Z));
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Normalize (vVector : TVertex) : TVertex;
//Returns a normalized vector (a vector of length 1)

var
  vMagnitude : Single;

begin
  //First, we get the magnitude of the vector
  vMagnitude := Magnitude (vVector);

  //Then we divide the vector by its own magnitude, to get a vector of length 1
  vVector.X := vVector.X / vMagnitude;
  vVector.Y := vVector.Y / vMagnitude;
  vVector.Z := vVector.Z / vMagnitude;

  Result := vVector;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Normal (vPoly : Array of TVertex) : TVertex;
//Returns the normal of a polygon (the direction the polygon is facing)

var
  vVector1, vVector2 : TVertex;
  vNormal : TVertex;

begin
  //First we get two vectors from the polygon
  vVector1 := Vector(vPoly[1], vPoly[0]);
  vVector2 := Vector(vPoly[2], vPoly[0]);

  //Next, we get the cross product of the 2 vectors, remember to input the vectors counter clockwise
  vNormal := Cross(vVector1,vVector2);

  //Now that we have the direction, it's still an unknown length, so we normalize it.
  vNormal := Normalize(vNormal);

  //Return our Normal
  Result := vNormal;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function Dot (vVector1, vVector2 : TVertex) : Single;
//Returns the dot product of two vectors

begin
  //There are two formulas for the dot product of two vectors :
  //A.B = (A.X*B.X) + (A.Y*B.Y) + (A.Z*B.Z)
  //A.B = ||A||*||B|| cos(theta)

  Result := (vVector1.X * vVector2.X) +
            (vVector1.Y * vVector2.Y) +
            (vVector1.Z * vVector2.Z);
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function AngleBetweenVectors(vVector1, vVector2 : TVertex) : Single;
//Returns the angle between two vectors

var
  DotProduct : Single;
  VectorsMagnitude : Single;
  Angle : Extended;

begin
  //First, we get the dot product of the two vectors
  DotProduct := Dot(vVector1,vVector2);

  //Then, we get the magnitude of the two vectors
  VectorsMagnitude := Magnitude(vVector1) * Magnitude(vVector2);

  //We divide the dot product by the magnitude, so then we are left with the cosine of the angle
  //between the two vectors
  Angle := arccos(DotProduct / VectorsMagnitude);

  if (isnan(Angle)) then
  begin
    Result := 0;
    exit;
  end;

  Result := Angle;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function ClosestPointOnLine (vA, vB, vPoint : TVertex) : TVertex;
//Returns the closest point on a line to a given point

var
  vVector1, vVector2, vVector3 : TVertex;
  vClosestPoint : TVertex;
  D, T : Single; 

begin
  //First, we create a vector from our end point vA to our point vPoint
  vVector1.X := vPoint.X - vA.X;
  vVector1.Y := vPoint.Y - vA.Y;
  vVector1.Z := vPoint.Z - vA.Z;

  //Now we create a normalized direction vector from end point vA to end point vB
  vVector2.X := vB.X - vA.X;
  vVector2.Y := vB.Y - vA.Y;
  vVector2.Z := vB.Z - vA.Z;
  vVector2 := Normalize (vVector2);

  //Now we use the distance formula to find the distance of the line segment
  D := Distance(vA, vB);

  //Using the dot product, we project the vVector1 onto the vector vVector2. This essentially
  //gives us the distance of our projected vector from vA
  T := Dot(vVector2, vVector1);

  //If our projected distance from vA, "t",  is greater than or equal to 0, it must be closest to the end point
  //vA.  So we return this end point.
  if (T<=0) then
  begin
    Result := vA;
    exit;
  end;

  //If our projected distance from vA, "t", is greater than or equal to the magnitude or distance of the line
  //segment, it must be closest to the end point vB, so we return vB.
  if (T >=D) then
  begin
    Result := vB;
    exit;
  end;

  //Here we create a vector that is of length T and in the direction of vVector2
  vVector3.X := vVector2.X * T;
  vVector3.Y := vVector2.Y * T;
  vVector3.Z := vVector2.Z * T;

  //To find the closest point on the line, we just add vVector3 to the original end point vA
  vClosestPoint.X := vA.X + vVector3.X;
  vClosestPoint.Y := vA.Y + vVector3.Y;
  vClosestPoint.Z := vA.Z + vVector3.Z;

  Result := vClosestPoint;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function PlaneDistance (vNormal, vPoint : TVertex) : Single;
//Returns the distance between a plane and the origin

var
  Distance : Single;    //Variable to hold the distance between a plane and the origin

begin
  //We use the plane equation (Ax + By + Cz + D = 0) to find the distance. We see that we do not have the value of D.
  //If we rearrange the equation a bit we get : (D = - (Ax + By +Cz))
  Distance := -((vNormal.X * vPoint.X) + (vNormal.Y * vPoint.Y) + (vNormal.Z * vPoint.Z));

  Result := Distance;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function IntersectionPoint(vNormal : TVertex; vLine : Array of TVertex; Distance : Single) : TVertex;
//Returns the intersection point between a line and a plane

var
  vLineDir : TVertex;   //Variable to hold the direction of our line
  vPoint : TVertex;
  Numerator, Denominator : Single;
  Dist : Single;

begin
  //First, we get a vector for our line and then we normalize it
  vLineDir := Vector (vLine[0], vLine[1]);
  vLineDir := Normalize(vLineDir);

  //Now we use the plane equation (Distance = Ax + By + Cz + D) to find the distance from one of our points to the
  //plane.  CHECK WHY WE NEGATE THE DISTANCE
  Numerator := - ((vNormal.X * vLine[0].X) +
                  (vNormal.Y * vLine[0].Y) +
                  (vNormal.Z * vLine[0].Z) + Distance);

  //If we take the dot product of the line and the normal, it will give us the cosine of the angle between the two,
  //since they are both normalized
  Denominator := Dot(vNormal,vLineDir);

  //CHECK WHY WE DO THIS
  if (Denominator = 0) then
  begin
    Result := vLine[0];
    exit;
  end;

  Dist := Numerator / Denominator;

  vPoint.X := (vLine[0].X + (vLineDir.X * Dist));
  vPoint.Y := (vLine[0].Y + (vLineDir.Y * Dist));
  vPoint.Z := (vLine[0].Z + (vLineDir.Z * Dist));

  Result := vPoint;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function InsidePolygon (vIntersection : TVertex; vPolygon : Array of TVertex; VertexCount : Integer) : Boolean;
//Checks to see if a point is in the bounds of a polygon

var
  Angle : Single;
  vA, vB : TVertex;
  i : Integer;

begin
  Angle := 0;
  
  //First, we create vectors from each of the vertexes to the intersection point.  We add up the angles
  //between these vectors.
  for i := 0 to VertexCount - 1 do
  begin
    vA := Vector(vPolygon[i],vIntersection);
    vB := Vector(vPolygon[(i+1) mod VertexCount], vIntersection);
    Angle := Angle + AngleBetweenVectors(vA,vB);
  end;

  //If the angle between all the vectors combined is greater than 360 degrees, it was in the polygon.
  if (Angle >= (MATCH_FACTOR * 2 * PI)) then
  begin
    Result := True;
    exit;
  end;

  Result := False;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function IntersectedPlane (vPolygon, vLine : Array of TVertex; var vNormal : TVertex; var OriginDistance : Single) : Boolean;
//Checks to see if a line intersects a plane

var
  Distance1, Distance2 : Single;

begin
  //First we get the normal of our plane
  vNormal := Normal(vPolygon);

  //Next, we get the distance from our plane to the origin
  OriginDistance := PlaneDistance (vNormal,vPolygon[0]);

  //Now we get the distances that the 2 points on the line are from the plane, using
  //the plane equation : Distance = Ax + By + Cz + D
  Distance1 := (vNormal.X*vLine[0].X) + (vNormal.Y*vLine[0].Y) + (vNormal.Z*vLine[0].Z) + OriginDistance;
  Distance2 := (vNormal.X*vLine[1].X) + (vNormal.Y*vLine[1].Y) + (vNormal.Z*vLine[1].Z) + OriginDistance;

  //Now, if the two points are on opposite sides of the plane, the distances will not have the same sign (+ and -).
  //Therefore, if we multiply the 2 sides and we get a negative result, we have collided
  if (Distance1 * Distance2 >= 0) then
  begin
    Result := False;
    exit;
  end;

  Result := True;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function IntersectedPolygon (vPolygon, vLine : Array of TVertex; VertexCount : Integer) : Boolean;
//Checks to see if a line intersects a polygon

var
  vNormal : TVertex;
  vIntersection : TVertex;
  OriginDistance : Single;

begin
  //First we check if the line intersects the plane.  If it's not, there's no need to go on ...
  if not (IntersectedPlane(vPolygon,vLine,vNormal,OriginDistance)) then
  begin
    Result := False;
    exit;
  end;

  //Now that we have the normal and origin distance passed back from IntersectedPlane, we can use it to
  //calculate the intersection point
  vIntersection := IntersectionPoint(vNormal,vLine,OriginDistance);

  //Now that we have the intersection point, we need to test if it's inside the polygon
  if (InsidePolygon(vIntersection,vPolygon,VertexCount)) then
  begin
    Result := True;
    exit;
  end;

  //If we get here, we haven't collided
  Result := False;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function ClassifySphere(var vCenter : TVertex; var vNormal : TVertex; var vPoint : TVertex; Radius : Single; var Distance : Single) : Integer;
//              ClassifySphere : This tells us if a sphere is in front, behind, or intersects a plane

var
  D : Single;

begin
  //First we need to find the distance of our polygon plane to the origin
  D := PlaneDistance(vNormal,vPoint);

  //Here we use the distance formula to get the distance that the center of our sphere is
  //from the plane
  Distance := ((vNormal.X*vCenter.X) +
               (vNormal.Y*vCenter.Y) +
               (vNormal.Z*vCenter.Z) + D);

  //If the distance is greater than the radius, then the sphere is intersecting the plane
  if abs(Distance) < Radius then
  begin
    Result := INTERSECTS;
    exit;
  end
  //If the distance is greater than or equal to the radius, it is completely in front of the plane
  else if Distance >= Radius then
  begin
    Result := FRONT;
    exit;
  end
  else
  begin
    Result := BEHIND;
    exit;
  end;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function EdgeSphereCollision(var vCenter : TVertex; vPolygon : Array of TVertex; VertexCount : Integer; Radius : Single) : Boolean;
//Checks if the sphere is intersecting any of the edges of the polygon

var
  vPoint : TVertex;
  i : Integer;
  D : Single;

begin
  for i := 0 to VertexCount - 1 do
  begin
    vPoint := ClosestPointOnLine(vPolygon[i],vPolygon[(i+1) mod VertexCount],vCenter);
    D := Distance(vCenter,vPoint);
    if (D < Radius) then
    begin
      Result := True;
      exit;
    end;
  end;
  Result := False;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function SpherePolygonCollision(vPolygon : Array of TVertex; var vCenter : TVertex; VertexCount : Integer; Radius : Single) : Boolean;
//Checks to see if a sphere intersects a polygon

var
  vNormal : TVertex;  //Variable for the normal of the polygon
  vOffset, vPosition : TVertex;
  Dist : Single;
  Classification : Integer;

begin
  //First we get the normal of the polygon
  vNormal := Normal(vPolygon);

  //Initialize our distance variable
  Dist := 0.0;

  //Next we check where the sphere is : in front, behind, or intersecting
  Classification := ClassifySphere(vCenter,vNormal,vPolygon[0], Radius,Dist);

  //If the sphere intersects, we carry on with our testing
  if (Classification = INTERSECTS) then
  begin
    //First, we get the offset to the plane
    vOffset.X := vNormal.X * Dist;
    vOffset.Y := vNormal.Y * Dist;
    vOffset.Z := vNormal.Z * Dist;

    //Then we just subtract it from the center of the sphere
    vPosition.X := vCenter.X - vOffset.X;
    vPosition.Y := vCenter.Y - vOffset.Y;
    vPosition.Z := vCenter.Z - vOffset.Z;

    //Then we check if it is inside the bounds of the polygon
    if (InsidePolygon(vPosition,vPolygon,VertexCount)) then
    begin
      Result := True;
      exit;
    end
    else
    begin
      if(EdgeSphereCollision(vCenter, vPolygon, vertexCount, radius)) then
			begin
				Result := True;
        exit;
			end;
    end;
  end;

  //If we get here, we obviously didn't collide
  Result := False;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

function GetCollisionOffset (var vNormal : TVertex; Radius, Dist : Single) : TVertex;
//Returns the offset to move the center of the sphere off the collided polygon

var
  vOffset : TVertex;
  DistOver : Single;

begin
  //Find the distance that our sphere is overlapping the plane, then find the direction vector to move our sphere
  DistOver := Radius - Dist;
  vOffset.X := vNormal.X * DistOver;
  vOffset.Y := vNormal.Y * DistOver;
  vOffset.Z := vNormal.Z * DistOver;

  Result := vOffset;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

end.

