unit glPower;

//=============================================
//
//    Power Tracker
//    Author: David "DavesLord" Caouette
//    Based on Win32 SDK documentation
//    Description: All that is related to power is here
//
//=============================================

interface

uses OpenGL, Windows, Messages;

const
  // APM messages
  PBT_APMQUERYSUSPEND       = $0000;
  PBT_APMQUERYSTANDBY       = $0001;
  PBT_APMQUERYSUSPENDFAILED = $0002;
  PBT_APMQUERYSTANDBYFAILED = $0003;
  PBT_APMSUSPEND            = $0004;
  PBT_APMSTANDBY            = $0005;
  PBT_APMRESUMECRITICAL     = $0006;
  PBT_APMRESUMESUSPEND      = $0007;
  PBT_APMRESUMESTANDBY      = $0008;
  PBT_APMBATTERYLOW         = $0009;
  PBT_APMPOWERSTATUSCHANGE  = $000A;
  PBT_APMOEMEVENT           = $000B;

  {AC_LINE_OFFLINE = 0;
  AC_LINE_ONLINE = 1;
  AC_LINE_BACKUP_POWER = 2;
  AC_LINE_UNKNOWN = 255;

  BATTERY_FLAG_HIGH = 1;
  BATTERY_FLAG_LOW = 2;
  BATTERY_FLAG_CRITICAL = 4;
  BATTERY_FLAG_CHARGING = 8;
  BATTERY_FLAG_NO_BATTERY = $80;
  BATTERY_FLAG_UNKNOWN = 255;
  BATTERY_PERCENTAGE_UNKNOWN = 255;
  BATTERY_LIFE_UNKNOWN = DWORD($FFFFFFFF);}

procedure LoadPowerTextures;
procedure RenderStatusChange;
procedure CheckStatusChange;

var
   ACBackupTex, ACOfflineTex, ACOnlineTex, ACUnknownTex: glUInt; // AC Line status textures
   BFChargTex, BFCritTex, BFLowTex, BFHighTex, BFNoBat, BFBatUnknown: glUInt; // Battery power status
   PWRPnlTop, PWRPnlSide, PWRPnlFront: glUInt; // Power Panel textures

   CurrentAC: glUInt;
   CurrentPWR: glUInt;
   BarWidth: glFloat;

implementation

uses glRender, Textures;

procedure LoadPowerTextures;
begin
   LoadTexture('lab\PWRPNLFT.jpg',     PWRPnlFront,   false);
   LoadTexture('lab\PWRPNL_S.jpg',     PWRPnlSide,    false);
   LoadTexture('lab\PWRPNL_T.jpg',     PWRPnlTop,     false);
   // Power Panel textures
   LoadTexture('lab\AC_BACKUP.jpg',    ACBackupTex,   false);
   LoadTexture('lab\AC_OFFLINE.jpg',   ACOfflineTex,  false);
   LoadTexture('lab\AC_ONLINE.jpg',    ACOnlineTex,   false);
   LoadTexture('lab\AC_UNKNOWN.jpg',   ACUnknownTex,  false);
   // AC Line Status
   LoadTexture('lab\CHARGINGPWR.jpg',  BFChargTex,    false);
   LoadTexture('lab\CRITICALPWR.jpg',  BFCritTex,     false);
   LoadTexture('lab\LOWPWR.jpg',       BFLowTex,      false);
   LoadTexture('lab\HIGHPWR.jpg',      BFHighTex,     false);
   LoadTexture('lab\NOBATTERYPWR.jpg', BFNoBat,       false);
   LoadTexture('lab\UNKNOWNPWR.jpg',   BFBatUnknown,  false);
   // Battery status
end;

procedure CheckStatusChange;
var Status: TSystemPowerStatus;
{ _SYSTEM_POWER_STATUS has 6 fields:
   ACLineStatus : Byte;       // 0 = OffLine, 1 = OnLine, 2 = Backup power (i.e: UPS), 255 = Unknown
   BatteryFlag : Byte;        // 1 = High, 2 = Low, 4 = Critical, 8 = Charging, 128 = No system battery, 255 = Unknown
   BatteryLifePercent : Byte; // 0 to 100 %, 255 if unknown
   Reserved1 : Byte;          // Reserved
   BatteryLifeTime : DWORD;      // LifeTime in Seconds (approx.), or $FFFFFFFF if unknown
   BatteryFullLifeTime : DWORD;  // FullLifeTime in Seconds (approx.), or $FFFFFFFF if unknown}
begin
   if GetSystemPowerStatus(Status) then
   // Here, we get our Power Status record filled
   begin
      case Status.ACLineStatus of
         AC_LINE_OFFLINE:        CurrentAC := ACOfflineTex;
         AC_LINE_ONLINE:         CurrentAC := ACOnlineTex;
         AC_LINE_BACKUP_POWER:   CurrentAC := ACBackupTex;
         AC_LINE_UNKNOWN:        CurrentAC := ACUnknownTex;
      end;

      if (Status.BatteryFlag and BATTERY_FLAG_LOW) = BATTERY_FLAG_LOW then
         CurrentPWR := BFLowTex;

      if (Status.BatteryFlag and BATTERY_FLAG_CRITICAL) = BATTERY_FLAG_CRITICAL then
         CurrentPWR := BFCritTex;

      if (Status.BatteryFlag and BATTERY_FLAG_HIGH) = BATTERY_FLAG_HIGH then
         CurrentPWR := BFHighTex;

      if (Status.BatteryFlag and BATTERY_FLAG_CHARGING) = BATTERY_FLAG_CHARGING then
         CurrentPWR := BFChargTex;                   

      if (Status.BatteryFlag and BATTERY_FLAG_NO_BATTERY) = BATTERY_FLAG_NO_BATTERY then
         CurrentPWR := BFNoBat;

      if (Status.BatteryFlag and BATTERY_FLAG_UNKNOWN) = BATTERY_FLAG_UNKNOWN then
         CurrentPWR := BFBatUnknown;
   end;

   if (Status.BatteryFlag and BATTERY_FLAG_UNKNOWN) <> BATTERY_FLAG_UNKNOWN then
      BarWidth := Abs((Status.BatteryLifePercent / 100) * 0.25)
   else
      BarWidth := 0.25;
end;

procedure RenderStatusChange;
begin   
   // ============ Power panel
   glBindTexture(GL_TEXTURE_2D, PWRPnlTop);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.25, 1.25, -15.5);
      glTexCoord(1,0);glVertex3f(-5.5,  1.25, -15.5);
      glTexCoord(1,1);glVertex3f(-5.5,  1.25, -15.0);
      glTexCoord(0,1);glVertex3f(-6.25, 1.25, -15.0);
   glEnd;

   glBindTexture(GL_TEXTURE_2D, PWRPnlTop);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.25, 0.25, -15.5);
      glTexCoord(1,0);glVertex3f(-5.5,  0.25, -15.5);
      glTexCoord(1,1);glVertex3f(-5.5,  0.25, -15.0);
      glTexCoord(0,1);glVertex3f(-6.25, 0.25, -15.0);
   glEnd;

   glBindTexture(GL_TEXTURE_2D, PWRPnlSide);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.25, 1.25, -15.5);
      glTexCoord(1,0);glVertex3f(-6.25, 1.25, -15.0);
      glTexCoord(1,1);glVertex3f(-6.25, 0.25, -15.0);
      glTexCoord(0,1);glVertex3f(-6.25, 0.25, -15.5);
   glEnd;

   {glDisable(GL_LIGHTING);
   // ============ Battery charge bar
   glBegin(GL_QUADS);
      glColor3f(0.0, 0.0, 1.0);glVertex3f(-6.195, 0.665, -11.99);
      glColor3f(0.0, 0.0, 1.0);glVertex3f(-6.195 + BarWidth, 0.665, -11.99);
      glColor3f(0.0, 0.0, 1.0);glVertex3f(-6.195 + BarWidth, 0.615,-11.99);
      glColor3f(0.0, 0.0, 1.0);glVertex3f(-6.195, 0.615, -11.99);
   glEnd;
   glEnable(GL_LIGHTING);}

   glCullFace(GL_BACK);

   glBindTexture(GL_TEXTURE_2D, PWRPnlSide);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-5.5, 1.25, -15.5);
      glTexCoord(1,0);glVertex3f(-5.5, 1.25, -15.0);
      glTexCoord(1,1);glVertex3f(-5.5, 0.25, -15.0);
      glTexCoord(0,1);glVertex3f(-5.5, 0.25, -15.5);
   glEnd;

   glBindTexture(GL_TEXTURE_2D, PWRPnlFront);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.25, 0.25, -15.0);
      glTexCoord(1,0);glVertex3f(-5.5,  0.25, -15.0);
      glTexCoord(1,1);glVertex3f(-5.5,  1.25, -15.0);
      glTexCoord(0,1);glVertex3f(-6.25, 1.25, -15.0);
   glEnd;
   // ============

   // ============ AC Line Status
   glBindTexture(GL_TEXTURE_2D, CurrentAC);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.20, 0.88, -14.99);
      glTexCoord(1,0);glVertex3f(-5.55, 0.88, -14.99);
      glTexCoord(1,1);glVertex3f(-5.55, 0.955, -14.99);
      glTexCoord(0,1);glVertex3f(-6.20, 0.955, -14.99);
   glEnd;
   // ============

   // ============ Battery Status
   glBindTexture(GL_TEXTURE_2D, CurrentPWR);
   glBegin(GL_QUADS);
      glTexCoord(0,0);glVertex3f(-6.20, 0.725, -14.99);
      glTexCoord(1,0);glVertex3f(-5.55, 0.725, -14.99);
      glTexCoord(1,1);glVertex3f(-5.55, 0.8, -14.99);
      glTexCoord(0,1);glVertex3f(-6.20, 0.8, -14.99);
   glEnd;
   // ============
   
   glCullFace(GL_FRONT);
end;

end.
