unit glShortcuts;

//=============================================
//
//    Shortcuts cubes
//    Author: David Caouette
//
//    Description: The cubic textured shortcuts
//
//=============================================

interface

uses OpenGL, Windows, Messages, SysUtils, glTypes, ShellAPI;

type

   TglShortcut = object
   public
      Position: TVertex;
      AppName: TFileName;
      Target: TFileName;
      Texture: glUInt;
      Description: PChar;
      ID: integer;
      ShortCutYTheta: single;
      Index: integer;
      procedure LoadShortCutTex(FileName: TFileName; iIndex: dWord);
      procedure RenderShortCut;
      procedure LoadCallShortCut;
      procedure Execute;
   end;

   PglShortCut = ^TglShortCut;

   TglShortcutList = array of TglShortCut;

procedure RenderShortCuts;
procedure LoadShortCuts(FileName: TFileName);
procedure SaveShortCuts(FileName: TFileName);

var
   glShortCutList: TglShortcutList;   
   ShortCutCount: integer;
   clShortCut: glUInt;

implementation

uses glRender, Textures, Graphics, INIFiles, Classes;

{ TglShortcut }

procedure LoadShortCuts(FileName: TFileName);
var
   i: integer;      
   ShortCutFile: TIniFile;
   glShortCut: TglShortCut;
begin
   DecimalSeparator := '.';
   ShortCutFile := TIniFile.Create(ExtractFilePath(ParamStr(0)) + 'SHORTCUT.INI');
   ShortCutCount := ShortCutFile.ReadInteger('GENERAL', 'ShortCutCount', 0);
   
   SetLength(glShortCutList, ShortCutCount);

   for i := 0 to ShortCutCount - 1 do
      with glShortCut do
      begin
         Position.X := ShortCutFile.ReadFloat(IntToStr(i), 'X', -8);
         Position.Y := ShortCutFile.ReadFloat(IntToStr(i), 'Y', 0.5);
         Position.Z := ShortCutFile.ReadFloat(IntToStr(i), 'Z', -8);
         AppName := ShortCutFile.ReadString(IntToStr(i), 'APPNAME', '');
         Target := ShortCutFile.ReadString(IntToStr(i), 'TARGET', '');
         Description := PChar(ShortCutFile.ReadString(IntToStr(i), 'DESC', ''));
         Index := ShortCutFile.ReadInteger(IntToStr(i), 'ICONIDX', 0);
         ID := i;
         Texture := 0;
	 try
	    LoadShortCutTex(ShortCutFile.ReadString(IntToStr(i), 'ICON', ''), Index);
	 except
	    raise Exception.Create('Shortcut '+IntToStr(i)+ ' is invalid.');
	 end;
	 glShortCutList[i] := glShortCut;
      end;

   ShortCutFile.Free;
end;

procedure SaveShortCuts(FileName: TFileName);
var
   i: integer;      
   ShortCutFile: TIniFile;
   glShortCut: TglShortCut;
begin
   DecimalSeparator := '.';
   ShortCutFile := TIniFile.Create(ExtractFilePath(ParamStr(0)) + 'SHORTCUT.INI');
   ShortCutFile.WriteInteger('GENERAL', 'ShortCutCount', ShortCutCount);

   for i := 0 to ShortCutCount - 1 do
      with glShortCut do
      begin
         ShortCutFile.WriteFloat(IntToStr(i), 'X', Position.X);
         ShortCutFile.WriteFloat(IntToStr(i), 'Y', Position.Y);
         ShortCutFile.WriteFloat(IntToStr(i), 'Z', Position.Z);
         ShortCutFile.WriteString(IntToStr(i), 'APPNAME', AppName);
         ShortCutFile.WriteString(IntToStr(i), 'TARGET', Target);	 
         ShortCutFile.WriteInteger(IntToStr(i), 'ICONIDX', Index);
      end;

   ShortCutFile.Free;
end;

procedure RenderShortCuts;
var
   i: integer;
begin
   for i := 0 to (ShortCutCount - 1) do
      glShortCutList[i].RenderShortCut;
end;

procedure TglShortcut.Execute;
begin
   ShellExecute(h_wnd,
    nil,
    PChar(AppName),
    nil,
    PChar(Target),
    SW_SHOWNORMAL);
end;

procedure TglShortcut.LoadCallShortCut;
const
   OffFactor = 0.15;
begin
   clShortCut := glGenLists(1);
   glNewList(clShortCut, GL_COMPILE);
      glDisable(GL_CULL_FACE);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_COLOR);        // Making it transparently shining
      glEnable(GL_BLEND);
      glFrontFace(GL_CW);
      glBegin(GL_QUADS);

         glTexCoord(0,0);glVertex3f(OffFactor, OffFactor, -OffFactor);
         glTexCoord(1,0);glVertex3f(OffFactor, OffFactor, OffFactor);
         glTexCoord(1,1);glVertex3f(-OffFactor, OffFactor, OffFactor);
         glTexCoord(0,1);glVertex3f(-OffFactor, OffFactor, -OffFactor); // Top

         glTexCoord(0,0);glVertex3f(OffFactor, -OffFactor, -OffFactor);
         glTexCoord(1,0);glVertex3f(OffFactor, -OffFactor, OffFactor);
         glTexCoord(1,1);glVertex3f(-OffFactor, -OffFactor, OffFactor);
         glTexCoord(0,1);glVertex3f(-OffFactor, -OffFactor, -OffFactor); // Bottom

         glTexCoord(0,0);glVertex3f(-OffFactor, -OffFactor, -OffFactor);
         glTexCoord(1,0);glVertex3f(-OffFactor, -OffFactor, OffFactor);
         glTexCoord(1,1);glVertex3f(-OffFactor, OffFactor, OffFactor);
         glTexCoord(0,1);glVertex3f(-OffFactor, OffFactor, -OffFactor); // Left

         glTexCoord(0,0);glVertex3f(OffFactor, -OffFactor, OffFactor);
         glTexCoord(1,0);glVertex3f(OffFactor, -OffFactor, -OffFactor);
         glTexCoord(1,1);glVertex3f(OffFactor, OffFactor, -OffFactor);
         glTexCoord(0,1);glVertex3f(OffFactor, OffFactor, OffFactor); // Right

         glTexCoord(0,0);glVertex3f(-OffFactor, -OffFactor, OffFactor);
         glTexCoord(1,0);glVertex3f(OffFactor, -OffFactor, OffFactor);
         glTexCoord(1,1);glVertex3f(OffFactor, OffFactor, OffFactor);
         glTexCoord(0,1);glVertex3f(-OffFactor, OffFactor, OffFactor); // Front

         glTexCoord(0,0);glVertex3f(OffFactor, -OffFactor, -OffFactor);
         glTexCoord(1,0);glVertex3f(-OffFactor, -OffFactor, -OffFactor);
         glTexCoord(1,1);glVertex3f(-OffFactor, OffFactor, -OffFactor);
         glTexCoord(0,1);glVertex3f(OffFactor, OffFactor, -OffFactor); // Back
         
      glEnd;
      glFrontFace(GL_CCW);
      glDisable(GL_BLEND);
      glEnable(GL_CULL_FACE);
   glEndList;
end;

procedure TglShortcut.LoadShortCutTex(FileName: TFileName; iIndex: dWord);
var
   AIcon: TIcon;
   ABitmap: TBitmap;
   ABitmapStream: TMemoryStream;
   FileExt: string;
   pData: Pointer;
   hbuf, hLib: THandle;
   dIndex: Word;
begin
   dIndex := iIndex;
   pData := nil;
   FileExt := ExtractFileExt(FileName);
   AIcon := TIcon.Create;   
   
   if UpperCase(FileExt) = '.ICO' then   
      AIcon.LoadFromFile(FileName)
   else
      if (UpperCase(FileExt) = '.EXE')  or (UpperCase(FileExt) = '.ICL') then
         AIcon.Handle := ExtractIcon(hInstance, PChar(FileName), iIndex)
      else
         if (UpperCase(FileExt) = '.DLL') then
         begin
            hLib := LoadLibrary(PChar(FileName));
            if hLib <> 0 then
               AIcon.Handle := ExtractIcon(hLib, PChar(FileName), iIndex);
            FreeLibrary(hLib);
         end
         else
            AIcon.Handle := ExtractAssociatedIcon(hInstance, PChar(AppName), dIndex);

   ABitmap := TBitmap.Create;
   ABitmap.PixelFormat := pf24Bit;
   ABitmap.Width  := AIcon.Width;
   ABitmap.Height := AIcon.Height;
   ABitmap.Canvas.Brush.Color := $00FF0033;
   ABitmap.Canvas.FillRect(Rect(0,0,ABitmap.Width,ABitmap.Height));
   ABitmap.Canvas.Draw(0,0, AIcon);

   ABitmapStream := TMemoryStream.Create;
   ABitmap.SaveToStream(ABitmapStream);
   ABitmap.SaveToFile('test.bmp');

   try
      hbuf := GlobalAlloc(GMEM_MOVEABLE, ABitmapStream.size);
      try
         pData := GlobalLock(hbuf);
         try
            Move(ABitmapStream.Memory^, pData^, ABitmapStream.size);
         finally
            GlobalUnlock(hbuf);
         end;
      except
         GlobalFree(hbuf);
         raise;
      end;
   finally
      ABitmapStream.Free;
   end;

   SwapRGB(pData, ABitmap.Width * ABitmap.Height);
   //Texture := CreateTexture(ABitmap.Width, ABitmap.Height, GL_RGB, pData);
   LoadTexture('test.bmp', Texture, false);
   ABitmap.Free;
   AIcon.Free;
   ShortCutYTheta := 0;
   LoadCallShortCut;
end;

procedure TglShortcut.RenderShortCut;
begin
   with Position do
   begin            
      glPushMatrix;
      glTranslatef(X, Y, Z);
      glRotatef(ShortCutYTheta, 0, -1, 0);
      ShortCutYTheta := ShortCutYTheta + 1;
      if ShortCutYTheta > 360 then
         ShortCutYTheta := ShortCutYTheta - 360;
      glBindTexture(GL_TEXTURE_2D, Texture);
      glCallList(clShortCut); // Render our cube shortcut           
      glPopMatrix;
   end;
end;



end.
